<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\Bonus;
use App\Models\BonusLedger;
use App\Models\Checkin;
use App\Models\User;
use App\Models\UserLedger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class GetBonusController extends Controller
{
    public function index()
    {
        $data = Bonus::where('status', 'active')->first();
        return view('app.main.bonus.index', compact('data'));
    }

    public function gift()
    {
        return view('app.main.gift.index');
    }

    public function submitBonusCode(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'bonus_code' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => 'Bonus code is required.'
            ]);
        }

        $code = $request->bonus_code;
        $bonus = Bonus::where('status', 'active')->first();
        $user = Auth::user();

        if (!$bonus) {
            return response()->json(['status' => 0, 'message' => 'No active bonus available.']);
        }

        if ($code !== $bonus->code) {
            return response()->json(['status' => 0, 'message' => 'Invalid bonus code.']);
        }

        $checkBonusUses = BonusLedger::where('bonus_id', $bonus->id)
            ->where('user_id', $user->id)
            ->first();

        if ($checkBonusUses) {
            return response()->json(['status' => 0, 'message' => 'You have already used this bonus.']);
        }

        if ($bonus->counter >= $bonus->set_service_counter) {
            return response()->json(['status' => 0, 'message' => 'Bonus quota is full.']);
        }

        // Credit user balance
        $user->balance += $bonus->amount;
        $user->save();

        // Log to UserLedger
        $ledger = new UserLedger();
        $ledger->user_id = $user->id;
        $ledger->reason = 'bonus';
        $ledger->perticulation = 'Bonus Code Redeemed: ' . $code;
        $ledger->amount = $bonus->amount;
        $ledger->debit = $bonus->amount;
        $ledger->status = 'approved';
        $ledger->date = now()->format('d-m-Y H:i');
        $ledger->save();

        // Update bonus counter
        $bonus->increment('counter');

        // Record bonus usage
        $bonusLedger = new BonusLedger();
        $bonusLedger->user_id = $user->id;
        $bonusLedger->bonus_id = $bonus->id;
        $bonusLedger->amount = $bonus->amount;
        $bonusLedger->bonus_code = $code;
        $bonusLedger->save();

        return response()->json([
            'status' => 1,
            'message' => 'Bonus received successfully: ' . price($bonus->amount)
        ]);
    }

    public function checkin(Request $request)
    {
        $user = Auth::user();
        $lastCheckin = Checkin::where('user_id', $user->id)->orderByDesc('id')->first();

        if (!$lastCheckin || now()->greaterThanOrEqualTo(Carbon::parse($lastCheckin->date))) {
            $amount = setting('checkin');

            // Record check-in
            $checkin = new Checkin();
            $checkin->user_id = $user->id;
            $checkin->date = now()->addHours(24);
            $checkin->amount = $amount;
            $checkin->save();

            // Update user balance
            $user->balance += $amount;
            $user->save();

            // Log to UserLedger
            $ledger = new UserLedger();
            $ledger->user_id = $user->id;
            $ledger->reason = 'checkin';
            $ledger->perticulation = 'Daily check-in bonus';
            $ledger->amount = $amount;
            $ledger->debit = $amount;
            $ledger->status = 'approved';
            $ledger->date = now()->format('d-m-Y H:i');
            $ledger->save();

            return redirect()->back()->with('success', 'You have received your daily check-in bonus.');
        }

        $remaining = Carbon::parse($lastCheckin->date)->diffForHumans();
        return redirect()->back()->with('error', 'Already checked in. Try again ' . $remaining);
    }

    public function checkin_ledger()
    {
        $checkins = Checkin::where('user_id', Auth::id())->orderByDesc('id')->get();
        return view('app.main.checkin-ledger', compact('checkins'));
    }

    public function preview()
    {
        $datas = BonusLedger::with(['bonus', 'user'])
            ->where('user_id', Auth::id())
            ->orderByDesc('id')
            ->get();

        return view('app.main.bonus.bonus-preview', compact('datas'));
    }
}
